"""
Modified from OpenAI Baselines code to work with multi-agent envs
"""
import numpy as np
from multiprocessing import Process, Pipe
from .vec_env import VecEnv, CloudpickleWrapper


def worker(remote, parent_remote, env_fn_wrapper):
    parent_remote.close()
    env = env_fn_wrapper.x()
    terminate = False
    while True:
        cmd, data = remote.recv()
        if cmd == 'step':
            state, reward, done, available_actions = env.step(data)
            remote.send((state, reward, done, available_actions))
        elif cmd == 'reset':
            state, available_actions = env.reset()
            terminate = False
            remote.send((state, available_actions))
        elif cmd == 'close':
            remote.close()
            break
        else:
            raise NotImplementedError


class SubprocVecEnv(VecEnv):
    def __init__(self, env_fns, spaces=None):
        """
        envs: list of gym environments to run in subprocesses
        """
        self.waiting = False
        self.closed = False
        nenvs = len(env_fns)
        self.remotes, self.work_remotes = zip(*[Pipe() for _ in range(nenvs)])
        self.ps = [Process(target=worker, args=(work_remote, remote, CloudpickleWrapper(env_fn))) for
                   (work_remote, remote, env_fn) in zip(self.work_remotes, self.remotes, env_fns)]
        for p in self.ps:
            p.daemon = True  # if the main process crashes, we should not cause things to hang
            p.start()
        for remote in self.work_remotes:
            remote.close()

        VecEnv.__init__(self, len(env_fns), None, None)

    def step_async(self, actions):
        for remote, action in zip(self.remotes, actions):
            remote.send(('step', action))
        self.waiting = True

    def step_wait(self):
        results = [remote.recv() for remote in self.remotes]
        self.waiting = False
        state, rews, dones, available_actions = zip(*results)
        return np.stack(state), np.stack(rews), np.stack(dones), np.stack(available_actions)

    def reset(self):
        for remote in self.remotes:
            remote.send(('reset', None))
        self.waiting = True
        results = [remote.recv() for remote in self.remotes]
        self.waiting = False
        state, available_actions = zip(*results)
        return np.stack(state), np.stack(available_actions)

    def close(self):
        if self.closed:
            return
        if self.waiting:
            for remote in self.remotes:
                remote.recv()
        for remote in self.remotes:
            remote.send(('close', None))
        for p in self.ps:
            p.join()
        self.closed = True

